﻿using Hl7.Fhir.Model;
using Hl7.Fhir.Serialization;
using log4net;
using Microsoft.Azure.KeyVault;
using Microsoft.IdentityModel.Clients.ActiveDirectory;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.CreateResponseService.Interface;
using VA.PPMS.IWS.Functions.Configuration.Interface;
using VA.PPMS.IWS.QueueService.Interface;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.CreateResponseService
{
    public class CreateResponseService : ICreateResponseService
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;
        private readonly IQueueService _queueService;
        private readonly IPpmsHelper _ppmsHelper;
        private readonly IPpmsContextHelper _contextHelper;

        public CreateResponseService(ILog logger, IIwsConfiguration configuration, IQueueService queueService, IPpmsHelper ppmsHelper, IPpmsContextHelper contextHelper)
        {
            _logger = logger;
            _configuration = configuration;
            _queueService = queueService;
            _ppmsHelper = ppmsHelper;
            _contextHelper = contextHelper;
        }

        public async Task<string> CreateResponse(DasMessage message)
        {
            _logger.Info($"@@@@ INFO - Start CreateResponseService.CreateResponse @@@@");

            try
            {
                var batchId = message.Content;

                // Create XML response file
                var response = await CreateResponseDocument(message);

                _logger.Info($"@@@@ INFO - End Main Create Response Service for BatchId: {batchId} @@@@");

                return response;
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was a problem with the CreateResponseService. @@@@", ex);
                throw new PpmsServiceException($"There was a problem with the CreateResponseService.", ex);
            }
        }

        public async Task<string> CreateProviderPayload(DasMessage message)
        {
            _logger.Info($"@@@@ INFO - Start CreateResponseService.CreateProviderPayload @@@@");

            try
            {
                // Create XML response file
                var response = await CreateProviderXmlDoc(message);

                _logger.Info($"@@@@ INFO - End Main Create Response Service for BatchId: {Utilities.ValidateGuid(message.Content)} @@@@");

                return response;
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was a problem with the CreateResponseService @@@@", ex);
                throw new PpmsServiceException($"There was a problem with the CreateResponseService.", ex);
            }
        }

        public async Task NotifyOfResponse(DasMessage message)
        {
            try
            {
                // Parse base64 string to extract guid
                var shortGuid = ShortGuid.Parse(message.Content);
                var batchId = shortGuid.ToGuid().ToString();

                var batch = await GetBatchById(batchId);
                if (batch == null) throw new PpmsServiceException("Unable to find batch record.");

                // Retrieve ReceiverId from associated network
                if (batch.ppms_vaprovidernetwork_batch_network == null) throw new PpmsServiceException("Unable to determine associated network.");

                // Set header values
                message.ConversationId = batch.ppms_conversationid;
                message.ResponseConversationId = batch.ppms_transactionid;
                message.ReceiverId = batch.ppms_vaprovidernetwork_batch_network.ppms_shorthand;

                // Get URL paths from configuration
                var baseUrl = await _configuration.GetPpmsResponseNotificationUriAsync(message.IsVaReceiver);
                var requestUri = await _configuration.GetPpmsResponsePostUriAsync(message.IsVaReceiver);

                _logger.Info($"Receiver: {message.ReceiverId}, {message.ResponseConversationId}");
                string result = "Success";
                if (message.IsVaReceiver)
                {
                    _logger.Info($"--- INFO: Post to PIE");
                    var payload = await CreateResponseDocument(message);
                    //var ccnFile = await CreateProviderXmlDoc(message);
                    //_logger.Info(ccnFile);
                    //result = await PostXmlToDas(message, payload, baseUrl, requestUri);
                }
                else
                {
                    // Create new guid when response to CCN/TW
                    var documentRef = await CreateDocumentReference(batch.ppms_conversationid);
                    //result = await PostToDas(message, documentRef, baseUrl, requestUri);
                }

                if (!string.IsNullOrEmpty(result))
                {
                    _logger.Info($"-- NotifyOfResponse - Updating batch: {message.ToString()}");
                    await _ppmsHelper.UpdateBatch(message, "Response notification sent.", (int)ppms_batch_StatusCode.ReceiverNotified);
                }
            }
            catch (Exception ex)
            {
                _logger.Error("CreateResponseService.NotifyOfResponse: Unable to process response.");
                throw new PpmsServiceException("CreateResponseService.NotifyOfResponse: Unable to process response.", ex);
            }
        }

        private async Task<string> PostXmlToDas(DasMessage message, string xmlContent, string baseUri, string requestUri)
        {
            var client = (HttpWebRequest)WebRequest.Create($"{baseUri}{requestUri}");
            client.ContentType = "application/xml";
            client.Method = "POST";
            client.Accept = "application/xml";
            // Set DAS headers
            client.Headers.Add("X-ConversationID", message.ConversationId);
            client.Headers.Add("X-RoutingSenderID", message.SenderId);
            client.Headers.Add("X-RoutingReceiverIDs", message.ReceiverId);
            client.Headers.Add("X-TransactionID", message.TransactionId);

            var usePieSecurity = Convert.ToBoolean(await _configuration.GetUsePieSecurity());

            if (usePieSecurity)
            {
                client.PreAuthenticate = true;
                client.Headers.Add("Authorization", "Bearer " + await GetToken2());
            }

            using (var streamWriter = new StreamWriter(client.GetRequestStream()))
            {
                streamWriter.Write(xmlContent);
                streamWriter.Flush();
                streamWriter.Close();
            }

            var httpResponse = (HttpWebResponse)(await client.GetResponseAsync());
            if (httpResponse != null) _logger.Info($"Response: {httpResponse.StatusCode} - {httpResponse.StatusDescription}");
            return "Success";
        }

        private async Task<string> GetToken2()
        {
            var authority = await _configuration.GetAzureAdAuthority();
            var resource = await _configuration.GetPieAppRegResource();
            var clientId = await _configuration.GetAddressValidationAppId();
            var clientSecret = await _configuration.GetAddressValidationSecret();

            var authContext = new AuthenticationContext(authority);
            var clientCred = new ClientCredential(clientId, clientSecret);
            var result = await authContext.AcquireTokenAsync(resource, clientCred);

            if (result == null) throw new InvalidOperationException("Failed to obtain the JWT token");

            return result.AccessToken;
        }

        private async Task<string> PostToDas(DasMessage message, DocumentReference content, string baseUri, string requestUri)
        {
            using (var client = await GetHttpClient())
            {
                client.BaseAddress = new Uri(baseUri);
                // Set DAS headers
                client.DefaultRequestHeaders.Accept.Clear();
                client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json+fhir"));
                client.DefaultRequestHeaders.Add("X-ConversationID", message.ResponseConversationId);
                client.DefaultRequestHeaders.Add("X-RoutingSenderID", message.SenderId);
                client.DefaultRequestHeaders.Add("X-RoutingReceiverIDs", message.ReceiverId);
                client.DefaultRequestHeaders.Add("X-TransactionID", message.TransactionId);

                var sout = new FhirJsonSerializer();
                var strDocRef = sout.SerializeToString(content);
                //_logger.Info($"DEBUG DocRef: {strDocRef}");
                var docRef = new StringContent(strDocRef, Encoding.Default, "application/json+fhir");

                docRef.Headers.ContentType.CharSet = string.Empty;

                var response = await client.PostAsync(requestUri, docRef);

                if (response.IsSuccessStatusCode) return "Success";

                throw new PpmsServiceException($"Error Posting Response Notification to DAS. The error is {response.StatusCode.ToString()}");
            }
        }

        private async Task<DocumentReference> CreateDocumentReference(string conversationId)
        {
            // Construct call back URL
            var documentPath = await GetDocumentPath(conversationId);
            var content = new List<DocumentReference.ContentComponent>
            {
                new DocumentReference.ContentComponent { Attachment = new Attachment { ContentType = "application/xml", Url = documentPath } }
            };

            // Set DocumentReference properties
            var docRef = new DocumentReference
            {
                Custodian = new ResourceReference("PPMS"),
                Created = DateTime.Now.ToString("s", System.Globalization.CultureInfo.InvariantCulture),
                Indexed = new DateTimeOffset(DateTime.Now),
                Status = DocumentReferenceStatus.Current,
                Content = content
            };

            return docRef;
        }

        private async Task<string> GetDocumentPath(string conversationId)
        {
            if (string.IsNullOrEmpty(conversationId))
            {
                _logger.Info($"@@@@ INFO - Parameter invalid for ConversationId = {conversationId} @@@@");
                throw new PpmsServiceException("Unable to create document path, parameter invalid");
            }

            //{ relative - path}/ Binary /[documentURN] ? transactionID ={ unique ID}
            var relativePathPattern = await _configuration.GetResponseDocumentPathPatternAsync();
            return string.Format(relativePathPattern, conversationId);
        }

        private async Task<ppms_batch> GetBatchById(string batchId)
        {
            var context = await _contextHelper.GetContextAsync();
            var batch = context.ppms_batchSet.FirstOrDefault(b => b.ppms_batchId == new Guid(batchId));

            LoadBatchProperties(context, batch);

            return batch;
        }

        public async Task<ppms_batch> GetBatchByConversationId(string conversationId, bool loadProperties = true)
        {
            var context = await _contextHelper.GetContextAsync();
            var batch = context.ppms_batchSet.FirstOrDefault(b => b.ppms_conversationid == conversationId);

            if (loadProperties) LoadBatchProperties(context, batch);

            return batch;
        }

        public async Task<ppms_batch> GetBatchByTransactionId(string transactionId, bool loadProperties = true)
        {
            var context = await _contextHelper.GetContextAsync();
            var batch = context.ppms_batchSet.FirstOrDefault(b => b.ppms_transactionid == transactionId);

            LoadBatchProperties(context, batch);

            return batch;
        }

        public async Task<IList<ppms_batchdetailresult>> GetBatchDetailResultsByConversationId(string conversationId)
        {
            var context = await _contextHelper.GetContextAsync();

            var results =
                from bdr in context.ppms_batchdetailresultSet
                join bd in context.ppms_batchdetailSet
                    on bdr.ppms_batchdetail.Id equals bd.ppms_batchdetailId
                join b in context.ppms_batchSet
                    on bd.ppms_batch.Id equals b.Id
                where b.ppms_conversationid == conversationId
                select bdr;

            return results.ToList();
        }


        private void LoadBatchProperties(PpmsContext context, ppms_batch batch)
        {
            if (batch == null) throw new PpmsServiceException("Batch record does not exist");

            context.LoadProperty(batch, new Relationship("ppms_batch_batchdetail_batch"));
            context.LoadProperty(batch, new Relationship("ppms_vaprovidernetwork_batch_network"));
        }

        public async Task<string> CreateResponseDocument(DasMessage queueMessage)
        {
            try
            {
                if (string.IsNullOrEmpty(queueMessage.ConversationId))
                {
                    _logger.Info($"@@@@ INFO - [CreateResponseDocument] Parameter invalid for item @@@@");
                    return string.Empty;
                }

                var conversationId = queueMessage.ConversationId;
                var batch = await GetBatchByConversationId(conversationId);

                var details = batch.ppms_batch_batchdetail_batch;
                IEnumerable<ppms_batchdetail> ppmsBatchdetails = details as ppms_batchdetail[] ?? details.ToArray();
                if (!ppmsBatchdetails.Any())
                {
                    _logger.Info($"@@@@ INFO - Details not found @@@@");
                    return string.Empty;
                }

                _logger.Info($"@@@@ INFO - Details found @@@@");

                var detailResults = await GetBatchDetailResultsByConversationId(conversationId);

                // Create XML doc
                var doc = new ProviderResponses
                {
                    ConversationId = conversationId,
                    ProviderResponse = new List<ProviderResponse>()
                };

                // Set status variables
                int i = 0;
                int chunkSize = 500;
                int size = ppmsBatchdetails.Count();
                var timer = new Stopwatch();
                timer.Start();

                IEnumerable<ppms_batchdetailresult> batchDetailResults = null;
                // Capture batch details
                foreach (var detail in ppmsBatchdetails)
                {
                    i++;

                    // Provider node
                    var provider = new ProviderResponse
                    {
                        ProviderId = detail.ppms_providerid,
                        Success = detail.GetAttributeValue<bool>("ppms_isvalid")
                    };

                    // Set correlation id, if appropriate
                    if (detail.ppms_provider != null)
                    {
                        provider.CorrelationId = detail.ppms_provider.Id.ToString();
                    }

                    // Retrieve batch detail results
                    if (detailResults != null && detailResults.Any())
                        batchDetailResults = detailResults.Where(x => x.ppms_batchdetail.Id == detail.Id);
                    else
                        batchDetailResults = null;

                    if (batchDetailResults != null)
                    {
                        // Capture batch detail results
                        IEnumerable<ppms_batchdetailresult> ppmsBatchdetailresults = batchDetailResults as ppms_batchdetailresult[] ?? batchDetailResults.ToArray();
                        if (ppmsBatchdetailresults != null && ppmsBatchdetailresults.Any())
                        {
                            // Initialize results list
                            provider.Results = new Results { Item = new List<Result>() };

                            foreach (var detailResult in ppmsBatchdetailresults)
                            {
                                var result = new Result
                                {
                                    Type = detailResult.ppms_entitytype,
                                    Id = detailResult.ppms_name,
                                    Success = detailResult.ppms_isvalid.HasValue && detailResult.ppms_isvalid.Value
                                };

                                if (result.Type == "ProviderService")
                                {
                                    result.CorrelationId = detailResult.Id.ToString();
                                }

                                if (!result.Success)
                                {
                                    result.Header = detailResult.ppms_result;
                                    result.Message = detailResult.ppms_message;
                                }

                                provider.Results.Item.Add(result);
                            }
                        }
                    }

                    // Add to list of providers
                    doc.ProviderResponse.Add(provider);

                    // Log status
                    if (i % chunkSize == 0 || i == size)
                    {
                        _logger.Info($"Processing response: {i} of {size} [{timer.Elapsed}]");
                    }
                }

                _logger.Info($"Response processing complete. [{timer.Elapsed}]");
                timer.Stop();

                // Create response packet
                return await ConvertResponseToXml(doc);
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                _logger.Error($"@@@@ CreateResponseService ERROR - Fault: {ex} @@@@", ex);
                throw;
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ CreateResponseService ERROR - Exception: {ex} @@@@", ex);
                throw;
            }
        }

        public async Task<string> CreateProviderXmlDoc(DasMessage queueMessage)
        {
            try
            {
                if (string.IsNullOrEmpty(queueMessage.Content))
                {
                    _logger.Info($"@@@@ INFO - Batch ID not provided. @@@@");
                    return string.Empty;
                }

                var shortGuid = ShortGuid.Parse(queueMessage.Content);
                var batchId = shortGuid.ToGuid().ToString();
                var batch = await GetBatchById(batchId);

                return await ExportBatchToXml(batch);
            }
            catch (Exception ex)
            {
                _logger.Info($"!!!! ERROR - Error occured created provider XML document. !!!!", ex);
            }

            return string.Empty;
        }

        private async Task<string> ConvertResponseToXml(ProviderResponses response)
        {
            var packet = await _configuration.GetSchemaProfileAsync(SchemaOptions.SchemaProfiles.Response);

            var prefix = packet.Prefix;
            var nameSpace = packet.Namespace;

            return Utilities.SerializeInstance(response, prefix, nameSpace);
        }

        private async Task<string> ExportBatchToXml(ppms_batch batch)
        {
            Context.Account account = null;

            if (batch.ppms_batch_batchdetail_batch == null) return string.Empty;

            var providers = new List<Provider>();

            foreach (var item in batch.ppms_batch_batchdetail_batch)
            {
                account = await GetProvider(item.ppms_provider.Id);
                if (account != null)
                {
                    // convert provider to XML
                    providers.Add(ToProvider(account));
                }
            }

            var xml = Utilities.SerializeInstance(providers, "p", "https://DNS.URL/exchange/ccn/1.0");
            xml = xml.Replace("ArrayOfProvider", "Providers");

            return xml;
        }

        private async Task<Context.Account> GetProvider(Guid providerId)
        {
            var context = await _contextHelper.GetContextAsync();
            var entity = context.AccountSet.FirstOrDefault(b => b.Id == providerId);
            if (entity == null) throw new PpmsServiceException("Batch record does not exist");

            LoadProviderProperties(entity, context);

            return entity;
        }

        protected void LoadProviderProperties(Entity entity, PpmsContext context)
        {
            if (entity == null || context == null) return;

            context.LoadProperty(entity, "ppms_account_ppms_provideridentifier_Provider");
            context.LoadProperty(entity, "ppms_account_ppms_providerservice");
            context.LoadProperty(entity, "ppms_account_ppms_boardcertification");
            context.LoadProperty(entity, "ppms_account_organizationauthorizedofficial");
            context.LoadProperty(entity, "ppms_account_ppms_otherprovideridentifier");
            context.LoadProperty(entity, "contact_customer_accounts");
            context.LoadProperty(entity, "ppms_account_providerlicensure");
            context.LoadProperty(entity, "ppms_account_ppms_othername");
            context.LoadProperty(entity, "ppms_account_ppms_providertaxonomy");
            context.LoadProperty(entity, "ppms_account_deascheduleprivilege");
            context.LoadProperty(entity, "ppms_account_providernetworkid");
        }

        private Provider ToProvider(Context.Account account)
        {
            if (account == null) return null;

            var provider = new Provider();

            // Map properties
            provider.Email = account.EMailAddress1;
            provider.Phone = account.Telephone1;
            provider.Fax = account.Fax;

            // Address
            var address = new ProviderData.Address();
            address.Address1 = account.Address1_Line1;
            address.Address2 = account.Address1_Line2;
            address.Address3 = account.Address1_Line3;
            address.City = account.Address1_City;
            address.State = account.Address1_StateOrProvince;
            address.PostalCode = account.Address1_PostalCode;
            address.County = account.Address1_County;

            provider.Address = address;

            // NPIs
            var npis = new List<Npi>();
            var npi = new Npi();
            npi.Number = account.ppms_ProviderIdentifier;
            provider.Npis = new Npis() { Item = npis };

            // Specialties
            if (account.ppms_account_ppms_providertaxonomy != null)
            {
                var specialties = new List<Taxonomy>();
                foreach (var item in account.ppms_account_ppms_providertaxonomy)
                {
                    specialties.Add(new Taxonomy() { CodedSpecialty = item.ppms_codedspecialty });
                }
                provider.Specialties = new Specialties() { Item = specialties };
            }

            // Name
            var individual = new Individual();
            var names = account.Name.Split(',');
            individual.FirstName = names[0].Trim();
            individual.LastName = names[1].Trim();
            individual.MiddleName = "";

            // DEA Numbers
            if (account.ppms_account_deascheduleprivilege != null)
            {
                var deaNumbers = new List<DeaSchedulePrivilege>();
                foreach (var item in account.ppms_account_deascheduleprivilege)
                {
                    deaNumbers.Add(new DeaSchedulePrivilege() { DeaNumber = item.ppms_deanumber });
                }
                individual.DeaNumbers = new DeaSchedulePrivileges() { Item = deaNumbers };
            }

            // Set provider type
            var providerType = new ProviderType();
            providerType.Item = individual;

            provider.Type = providerType;

            return provider;
        }

        #region Certs

        public async Task<HttpClient> GetHttpClient()
        {
            await Task.Run(() => { });
            var clientHandler = new HttpClientHandler();
            clientHandler.ClientCertificates.Add(await GetCertKeyVault());

            return new HttpClient(clientHandler);
        }

        private async Task<X509Certificate2> GetCertKeyVault()
        {
            var keyVaultClient = new KeyVaultClient(GetToken);
            var result = await keyVaultClient.GetSecretAsync(await _configuration.GetDasCertUrl());

            if (result == null) throw new InvalidOperationException("Failed to obtain the certificate from Key Vault");

            var secret = Convert.FromBase64String(result.Value);
            return new X509Certificate2(secret, (string)null);
        }

        private async Task<string> GetToken(string authority, string resource, string scope)
        {
            var authContext = new AuthenticationContext(authority);
            var clientCred = new ClientCredential(await _configuration.GetDasAppId(), await _configuration.GetDasSecret());
            var result = await authContext.AcquireTokenAsync(resource, clientCred);

            if (result == null) throw new InvalidOperationException("Failed to obtain the token to retrieve certificate");

            return result.AccessToken;
        }

        #endregion
    }
}